﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.ServiceModel;
using System.Xml.Linq;
using VA.PPMS.Context;
using VA.PPMS.CRM.Plugins.Data;
using VA.PPMS.CRM.Plugins.Helper;

namespace VA.PPMS.CRM.Plugins
{
    public class BatchUpdate : IPlugin
    {
        private ITracingService _tracingService;
        private IOrganizationService _service;

        private const string PluginName = "BatchUpdate";
        private static string StorageAccountKey = "";

        public BatchUpdate(string unsecure, string secure)
        {
            //string xml = "<settings><SPCode>spc</SPCode><StorageAccountKey>sak</StorageAccountKey></settings>";
            StorageAccountKey = SettingsHelper.GetValueFromKey(secure, "StorageAccountKey");
        }

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            _tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                _tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity entity = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (entity.LogicalName != "ppms_batch")
                    return;

                _tracingService.Trace("Entity found");

                // Verify status scenario
                var status = entity.GetAttributeValue<OptionSetValue>("statuscode");
                if (status != null && status.Value != (int)PpmsHelper.Batch_StatusCode.ValidationComplete)
                {
                    _tracingService.Trace("Target status not set");
                    return;
                }

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                _service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    // Retrieve settings
                    var settings = SettingsHelper.GetSettings(_service);
                    if (!settings.IsValid)
                    {
                        _tracingService.Trace("ERROR: Unable to retrieve service settings. {0}", settings.ErrorMessage);
                        return;
                    }
                    settings.StorageAccountKey = StorageAccountKey;

                    _tracingService.Trace("Submit to queue...");
                    // Convert GUID to base64 string
                    var shortGuid = new ShortGuid(entity.Id);
                    FabricHelper.SubmitToQueue(settings, shortGuid.ToBase64());

                    // Update batch status code
                    var request = PpmsHelper.GetStateRequest(entity, (int)ppms_batchState.Active, (int)ppms_batch_StatusCode.RequestInitiated);
                    if (request != null) _service.Execute(request);

                    // Save response XML for batch records
                    //_tracingService.Trace("Saving response to batch...");
                    //SaveResponse(entity);
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    _tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    _tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            _tracingService.Trace("Done");
        }
    }
}
